'use client';

import React, { useState, useEffect } from 'react';
import {
  ArrowLeft,
  Save,
  Shield,
  Lock,
  Eye,
  EyeOff,
  Bell,
  Mail,
  Users,
  AlertCircle,
  Settings as SettingsIcon,
  Clock,
  CheckCircle
} from 'lucide-react';
import Link from 'next/link';
import { useRouter, useParams } from 'next/navigation';
import type { Department } from '@/types';
import { departmentApi } from '@/services/api';

interface DepartmentSettings {
  // Erişim ve Güvenlik
  requireApprovalForTickets: boolean;
  allowSelfAssignment: boolean;
  restrictViewToOwnTickets: boolean;

  // Bildirimler
  emailNotifications: boolean;
  smsNotifications: boolean;
  notifyOnNewTicket: boolean;
  notifyOnTicketUpdate: boolean;
  notifyOnTicketCompletion: boolean;

  // Otomatik İşlemler
  autoAssignTickets: boolean;
  autoAssignMethod: 'round_robin' | 'least_busy' | 'random';
  autoCloseCompletedTickets: boolean;
  autoCloseDelayHours: number;

  // Çalışma Saatleri
  enforceWorkingHours: boolean;
  workingHoursStart: string;
  workingHoursEnd: string;
  workingDays: string[];
}

export default function DepartmentSettingsPage() {
  const router = useRouter();
  const params = useParams();
  const departmentId = params?.id as string;

  const [department, setDepartment] = useState<Department | null>(null);
  const [loading, setLoading] = useState(false);
  const [loadingData, setLoadingData] = useState(true);

  const [settings, setSettings] = useState<DepartmentSettings>({
    // Varsayılan ayarlar
    requireApprovalForTickets: false,
    allowSelfAssignment: true,
    restrictViewToOwnTickets: false,

    emailNotifications: true,
    smsNotifications: false,
    notifyOnNewTicket: true,
    notifyOnTicketUpdate: false,
    notifyOnTicketCompletion: true,

    autoAssignTickets: false,
    autoAssignMethod: 'round_robin',
    autoCloseCompletedTickets: false,
    autoCloseDelayHours: 24,

    enforceWorkingHours: false,
    workingHoursStart: '09:00',
    workingHoursEnd: '18:00',
    workingDays: ['monday', 'tuesday', 'wednesday', 'thursday', 'friday']
  });

  useEffect(() => {
    if (departmentId) {
      fetchDepartment();
      fetchSettings();
    }
  }, [departmentId]);

  const fetchDepartment = async () => {
    try {
      const data = await departmentApi.getDepartment(departmentId);
      setDepartment({
        id: data.id,
        name: data.name,
        description: data.description || '',
        color: data.color,
        parentDepartmentId: data.parent_department_id,
        managerId: data.manager_id,
        isActive: Boolean(data.is_active),
        userCount: data.user_count || 0,
        createdAt: new Date(data.created_at),
        updatedAt: new Date(data.updated_at)
      });
    } catch (error) {
      console.error('Error fetching department:', error);
    }
  };

  const fetchSettings = async () => {
    try {
      setLoadingData(true);
      // TODO: Implement department settings API endpoint
      // const data = await departmentApi.getDepartmentSettings(departmentId);
      // setSettings(data);
    } catch (error) {
      console.error('Error fetching settings:', error);
    } finally {
      setLoadingData(false);
    }
  };

  const handleSettingChange = (key: keyof DepartmentSettings, value: any) => {
    setSettings(prev => ({ ...prev, [key]: value }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    try {
      setLoading(true);

      // TODO: Implement department settings update API endpoint
      // await departmentApi.updateDepartmentSettings(departmentId, settings);

      alert('Departman ayarları başarıyla güncellendi');
      router.push(`/departments/${departmentId}`);
    } catch (error) {
      console.error('Error updating settings:', error);
      alert('Ayarlar güncellenirken bir hata oluştu. Lütfen tekrar deneyin.');
    } finally {
      setLoading(false);
    }
  };

  const workingDayOptions = [
    { value: 'monday', label: 'Pazartesi' },
    { value: 'tuesday', label: 'Salı' },
    { value: 'wednesday', label: 'Çarşamba' },
    { value: 'thursday', label: 'Perşembe' },
    { value: 'friday', label: 'Cuma' },
    { value: 'saturday', label: 'Cumartesi' },
    { value: 'sunday', label: 'Pazar' }
  ];

  const toggleWorkingDay = (day: string) => {
    setSettings(prev => ({
      ...prev,
      workingDays: prev.workingDays.includes(day)
        ? prev.workingDays.filter(d => d !== day)
        : [...prev.workingDays, day]
    }));
  };

  return (
    <div className="p-6">
      {/* Header */}
      <div className="flex items-center justify-between mb-6">
        <div className="flex items-center space-x-4">
          <Link
            href={`/departments/${departmentId}`}
            className="flex items-center text-gray-600 hover:text-gray-900"
          >
            <ArrowLeft className="h-5 w-5 mr-2" />
            Departmana Dön
          </Link>
          <div className="h-6 border-l border-gray-300"></div>
          {department && (
            <div className="flex items-center">
              <div className={`w-4 h-4 rounded-full ${department.color} mr-2`}></div>
              <h1 className="text-2xl font-bold text-gray-900">{department.name} - Ayarlar</h1>
            </div>
          )}
        </div>
      </div>

      <form onSubmit={handleSubmit} className="max-w-4xl">
        {/* Erişim ve Güvenlik */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 mb-6">
          <h2 className="text-lg font-medium text-gray-900 flex items-center mb-4">
            <Shield className="h-5 w-5 mr-2 text-blue-600" />
            Erişim ve Güvenlik
          </h2>

          <div className="space-y-4">
            <label className="flex items-start">
              <input
                type="checkbox"
                checked={settings.requireApprovalForTickets}
                onChange={(e) => handleSettingChange('requireApprovalForTickets', e.target.checked)}
                className="mt-1 h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
              />
              <div className="ml-3">
                <span className="text-sm font-medium text-gray-700">Bilet İşlemleri için Onay Gereksin</span>
                <p className="text-xs text-gray-500 mt-1">
                  Bu departmandaki kullanıcılar bilet işlemlerini tamamlamak için yönetici onayı almalıdır
                </p>
              </div>
            </label>

            <label className="flex items-start">
              <input
                type="checkbox"
                checked={settings.allowSelfAssignment}
                onChange={(e) => handleSettingChange('allowSelfAssignment', e.target.checked)}
                className="mt-1 h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
              />
              <div className="ml-3">
                <span className="text-sm font-medium text-gray-700">Kendi Kendine Atama İzni</span>
                <p className="text-xs text-gray-500 mt-1">
                  Kullanıcılar kendilerine bilet/görev atayabilir
                </p>
              </div>
            </label>

            <label className="flex items-start">
              <input
                type="checkbox"
                checked={settings.restrictViewToOwnTickets}
                onChange={(e) => handleSettingChange('restrictViewToOwnTickets', e.target.checked)}
                className="mt-1 h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
              />
              <div className="ml-3">
                <span className="text-sm font-medium text-gray-700">Sadece Kendi Biletlerini Görüntüleme</span>
                <p className="text-xs text-gray-500 mt-1">
                  Kullanıcılar sadece kendilerine atanmış biletleri görebilir
                </p>
              </div>
            </label>
          </div>
        </div>

        {/* Bildirimler */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 mb-6">
          <h2 className="text-lg font-medium text-gray-900 flex items-center mb-4">
            <Bell className="h-5 w-5 mr-2 text-purple-600" />
            Bildirimler
          </h2>

          <div className="space-y-4">
            <div className="grid grid-cols-2 gap-4">
              <label className="flex items-center">
                <input
                  type="checkbox"
                  checked={settings.emailNotifications}
                  onChange={(e) => handleSettingChange('emailNotifications', e.target.checked)}
                  className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                />
                <Mail className="h-4 w-4 mx-2 text-gray-400" />
                <span className="text-sm text-gray-700">E-posta Bildirimleri</span>
              </label>

              <label className="flex items-center">
                <input
                  type="checkbox"
                  checked={settings.smsNotifications}
                  onChange={(e) => handleSettingChange('smsNotifications', e.target.checked)}
                  className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                />
                <span className="text-sm text-gray-700 ml-2">SMS Bildirimleri</span>
              </label>
            </div>

            <div className="border-t border-gray-200 pt-4 mt-4">
              <p className="text-sm font-medium text-gray-700 mb-3">Bildirim Türleri</p>
              <div className="space-y-2">
                <label className="flex items-center">
                  <input
                    type="checkbox"
                    checked={settings.notifyOnNewTicket}
                    onChange={(e) => handleSettingChange('notifyOnNewTicket', e.target.checked)}
                    className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                  />
                  <span className="text-sm text-gray-700 ml-3">Yeni bilet atandığında</span>
                </label>

                <label className="flex items-center">
                  <input
                    type="checkbox"
                    checked={settings.notifyOnTicketUpdate}
                    onChange={(e) => handleSettingChange('notifyOnTicketUpdate', e.target.checked)}
                    className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                  />
                  <span className="text-sm text-gray-700 ml-3">Bilet güncellendiğinde</span>
                </label>

                <label className="flex items-center">
                  <input
                    type="checkbox"
                    checked={settings.notifyOnTicketCompletion}
                    onChange={(e) => handleSettingChange('notifyOnTicketCompletion', e.target.checked)}
                    className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                  />
                  <span className="text-sm text-gray-700 ml-3">Bilet tamamlandığında</span>
                </label>
              </div>
            </div>
          </div>
        </div>

        {/* Otomatik İşlemler */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 mb-6">
          <h2 className="text-lg font-medium text-gray-900 flex items-center mb-4">
            <SettingsIcon className="h-5 w-5 mr-2 text-green-600" />
            Otomatik İşlemler
          </h2>

          <div className="space-y-4">
            <label className="flex items-start">
              <input
                type="checkbox"
                checked={settings.autoAssignTickets}
                onChange={(e) => handleSettingChange('autoAssignTickets', e.target.checked)}
                className="mt-1 h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
              />
              <div className="ml-3 flex-1">
                <span className="text-sm font-medium text-gray-700">Otomatik Bilet Atama</span>
                <p className="text-xs text-gray-500 mt-1">
                  Yeni biletler otomatik olarak müsait kullanıcılara atansın
                </p>

                {settings.autoAssignTickets && (
                  <div className="mt-3">
                    <label className="block text-xs font-medium text-gray-700 mb-1">
                      Atama Yöntemi
                    </label>
                    <select
                      value={settings.autoAssignMethod}
                      onChange={(e) => handleSettingChange('autoAssignMethod', e.target.value)}
                      className="w-full px-3 py-2 text-sm border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                    >
                      <option value="round_robin">Sırayla (Round Robin)</option>
                      <option value="least_busy">En Az Yoğun Kullanıcı</option>
                      <option value="random">Rastgele</option>
                    </select>
                  </div>
                )}
              </div>
            </label>

            <label className="flex items-start">
              <input
                type="checkbox"
                checked={settings.autoCloseCompletedTickets}
                onChange={(e) => handleSettingChange('autoCloseCompletedTickets', e.target.checked)}
                className="mt-1 h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
              />
              <div className="ml-3 flex-1">
                <span className="text-sm font-medium text-gray-700">Tamamlanan Biletleri Otomatik Kapat</span>
                <p className="text-xs text-gray-500 mt-1">
                  Tamamlanan biletler belirli süre sonra otomatik kapansın
                </p>

                {settings.autoCloseCompletedTickets && (
                  <div className="mt-3">
                    <label className="block text-xs font-medium text-gray-700 mb-1">
                      Kapatma Gecikmesi (Saat)
                    </label>
                    <input
                      type="number"
                      min="1"
                      max="168"
                      value={settings.autoCloseDelayHours}
                      onChange={(e) => handleSettingChange('autoCloseDelayHours', parseInt(e.target.value))}
                      className="w-32 px-3 py-2 text-sm border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                    />
                  </div>
                )}
              </div>
            </label>
          </div>
        </div>

        {/* Çalışma Saatleri */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 mb-6">
          <h2 className="text-lg font-medium text-gray-900 flex items-center mb-4">
            <Clock className="h-5 w-5 mr-2 text-orange-600" />
            Çalışma Saatleri
          </h2>

          <div className="space-y-4">
            <label className="flex items-start">
              <input
                type="checkbox"
                checked={settings.enforceWorkingHours}
                onChange={(e) => handleSettingChange('enforceWorkingHours', e.target.checked)}
                className="mt-1 h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
              />
              <div className="ml-3 flex-1">
                <span className="text-sm font-medium text-gray-700">Çalışma Saatleri Kontrolü</span>
                <p className="text-xs text-gray-500 mt-1">
                  Belirli çalışma saatleri dışında otomatik atama yapılmasın
                </p>

                {settings.enforceWorkingHours && (
                  <div className="mt-4 space-y-4">
                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <label className="block text-xs font-medium text-gray-700 mb-1">
                          Başlangıç Saati
                        </label>
                        <input
                          type="time"
                          value={settings.workingHoursStart}
                          onChange={(e) => handleSettingChange('workingHoursStart', e.target.value)}
                          className="w-full px-3 py-2 text-sm border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                        />
                      </div>
                      <div>
                        <label className="block text-xs font-medium text-gray-700 mb-1">
                          Bitiş Saati
                        </label>
                        <input
                          type="time"
                          value={settings.workingHoursEnd}
                          onChange={(e) => handleSettingChange('workingHoursEnd', e.target.value)}
                          className="w-full px-3 py-2 text-sm border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                        />
                      </div>
                    </div>

                    <div>
                      <label className="block text-xs font-medium text-gray-700 mb-2">
                        Çalışma Günleri
                      </label>
                      <div className="grid grid-cols-3 gap-2">
                        {workingDayOptions.map((day) => (
                          <label
                            key={day.value}
                            className={`flex items-center justify-center px-3 py-2 rounded-lg border-2 cursor-pointer transition-all ${
                              settings.workingDays.includes(day.value)
                                ? 'border-blue-500 bg-blue-50 text-blue-700'
                                : 'border-gray-200 hover:border-gray-300'
                            }`}
                          >
                            <input
                              type="checkbox"
                              checked={settings.workingDays.includes(day.value)}
                              onChange={() => toggleWorkingDay(day.value)}
                              className="sr-only"
                            />
                            <span className="text-sm font-medium">{day.label}</span>
                          </label>
                        ))}
                      </div>
                    </div>
                  </div>
                )}
              </div>
            </label>
          </div>
        </div>

        {/* Submit Button */}
        <div className="flex justify-end space-x-3">
          <Link
            href={`/departments/${departmentId}`}
            className="px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50"
          >
            İptal
          </Link>
          <button
            type="submit"
            disabled={loading}
            className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"
          >
            {loading ? (
              <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
            ) : (
              <Save className="h-4 w-4 mr-2" />
            )}
            {loading ? 'Kaydediliyor...' : 'Ayarları Kaydet'}
          </button>
        </div>
      </form>
    </div>
  );
}
